import tkinter as tk
from tkinter import ttk
import math
import time
from threading import Thread

class BallisticMotion:
    def __init__(self, root):
        self.root = root
        self.root.title("Баллистическое движение")
        self.root.geometry("1200x700")
        
        # Параметры симуляции
        self.initial_velocity = 50  # м/с
        self.initial_height = 0     # м
        self.launch_angle = 45      # градусы
        self.gravity = 9.81         # м/с²
        
        # Состояние симуляции
        self.simulating = False
        self.time_elapsed = 0
        self.max_height = 0
        self.max_range = 0
        self.total_time = 0
        
        # Траектория
        self.trajectory = []
        
        # Масштабирование
        self.scale = 5  # пикселей на метр
        self.ground_level = 500
        
        self.setup_ui()
        self.calculate_trajectory()
        self.draw_scene()
    
    def setup_ui(self):
        # Основной фрейм
        main_frame = ttk.Frame(self.root, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # === Левая панель - управление ===
        left_frame = ttk.Frame(main_frame, width=250)
        left_frame.pack(side=tk.LEFT, fill=tk.Y, padx=(0, 10))
        left_frame.pack_propagate(False)
        
        ttk.Label(left_frame, text="Параметры запуска", 
                 font=("Arial", 12, "bold")).pack(anchor=tk.W, pady=(0, 10))
        
        # Начальная скорость
        velocity_frame = ttk.LabelFrame(left_frame, text="Начальная скорость", padding="5")
        velocity_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(velocity_frame, text="Скорость (м/с):").pack(anchor=tk.W)
        self.velocity_var = tk.DoubleVar(value=self.initial_velocity)
        velocity_scale = ttk.Scale(velocity_frame, from_=10, to=100, 
                                  variable=self.velocity_var, orient=tk.HORIZONTAL)
        velocity_scale.pack(fill=tk.X, pady=2)
        
        self.velocity_value = ttk.Label(velocity_frame, text=f"{self.initial_velocity} м/с")
        self.velocity_value.pack(anchor=tk.W)
        
        velocity_scale.configure(command=self.on_velocity_change)
        
        # Угол запуска
        angle_frame = ttk.LabelFrame(left_frame, text="Угол запуска", padding="5")
        angle_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(angle_frame, text="Угол (градусы):").pack(anchor=tk.W)
        self.angle_var = tk.DoubleVar(value=self.launch_angle)
        angle_scale = ttk.Scale(angle_frame, from_=0, to=90, 
                               variable=self.angle_var, orient=tk.HORIZONTAL)
        angle_scale.pack(fill=tk.X, pady=2)
        
        self.angle_value = ttk.Label(angle_frame, text=f"{self.launch_angle}°")
        self.angle_value.pack(anchor=tk.W)
        
        angle_scale.configure(command=self.on_angle_change)
        
        # Начальная высота
        height_frame = ttk.LabelFrame(left_frame, text="Начальная высота", padding="5")
        height_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(height_frame, text="Высота (м):").pack(anchor=tk.W)
        self.height_var = tk.DoubleVar(value=self.initial_height)
        height_scale = ttk.Scale(height_frame, from_=0, to=100, 
                                variable=self.height_var, orient=tk.HORIZONTAL)
        height_scale.pack(fill=tk.X, pady=2)
        
        self.height_value = ttk.Label(height_frame, text=f"{self.initial_height} м")
        self.height_value.pack(anchor=tk.W)
        
        height_scale.configure(command=self.on_height_change)
        
        # Ускорение свободного падения
        gravity_frame = ttk.LabelFrame(left_frame, text="Гравитация", padding="5")
        gravity_frame.pack(fill=tk.X, pady=5)
        
        ttk.Label(gravity_frame, text="g (м/с²):").pack(anchor=tk.W)
        self.gravity_var = tk.DoubleVar(value=self.gravity)
        gravity_scale = ttk.Scale(gravity_frame, from_=1, to=20, 
                                 variable=self.gravity_var, orient=tk.HORIZONTAL)
        gravity_scale.pack(fill=tk.X, pady=2)
        
        self.gravity_value = ttk.Label(gravity_frame, text=f"{self.gravity} м/с²")
        self.gravity_value.pack(anchor=tk.W)
        
        gravity_scale.configure(command=self.on_gravity_change)
        
        # Управление симуляцией
        control_frame = ttk.LabelFrame(left_frame, text="Управление", padding="5")
        control_frame.pack(fill=tk.X, pady=5)
        
        self.start_button = ttk.Button(control_frame, text="Запуск", 
                                      command=self.start_simulation)
        self.start_button.pack(fill=tk.X, pady=2)
        
        self.stop_button = ttk.Button(control_frame, text="Стоп", 
                                     command=self.stop_simulation, state=tk.DISABLED)
        self.stop_button.pack(fill=tk.X, pady=2)
        
        self.reset_button = ttk.Button(control_frame, text="Сброс", 
                                      command=self.reset_simulation)
        self.reset_button.pack(fill=tk.X, pady=2)
        
        # Теоретические значения
        theory_frame = ttk.LabelFrame(left_frame, text="Теоретические значения", padding="5")
        theory_frame.pack(fill=tk.X, pady=5)
        
        self.max_height_label = ttk.Label(theory_frame, text="Макс. высота: 0 м", 
                                         font=("Arial", 9))
        self.max_height_label.pack(anchor=tk.W)
        
        self.max_range_label = ttk.Label(theory_frame, text="Дальность полета: 0 м", 
                                        font=("Arial", 9))
        self.max_range_label.pack(anchor=tk.W)
        
        self.total_time_label = ttk.Label(theory_frame, text="Время полета: 0 с", 
                                         font=("Arial", 9))
        self.total_time_label.pack(anchor=tk.W)
        
        # === Центральная панель - визуализация ===
        center_frame = ttk.Frame(main_frame)
        center_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=10)
        
        # Холст для отрисовки
        self.canvas = tk.Canvas(center_frame, bg="lightblue", highlightthickness=1, 
                               highlightbackground="gray")
        self.canvas.pack(fill=tk.BOTH, expand=True)
        
        # === Правая панель - текущие значения ===
        right_frame = ttk.Frame(main_frame, width=250)
        right_frame.pack(side=tk.RIGHT, fill=tk.Y, padx=(10, 0))
        right_frame.pack_propagate(False)
        
        ttk.Label(right_frame, text="Текущие значения", 
                 font=("Arial", 12, "bold")).pack(anchor=tk.W, pady=(0, 10))
        
        # Текущая скорость
        speed_frame = ttk.LabelFrame(right_frame, text="Скорость", padding="5")
        speed_frame.pack(fill=tk.X, pady=5)
        
        self.vx_label = ttk.Label(speed_frame, text="Vx: 0.00 м/с", 
                                 font=("Arial", 10))
        self.vx_label.pack(anchor=tk.W)
        
        self.vy_label = ttk.Label(speed_frame, text="Vy: 0.00 м/с", 
                                 font=("Arial", 10))
        self.vy_label.pack(anchor=tk.W)
        
        self.v_label = ttk.Label(speed_frame, text="V: 0.00 м/с", 
                                font=("Arial", 10, "bold"))
        self.v_label.pack(anchor=tk.W)
        
        # Положение
        position_frame = ttk.LabelFrame(right_frame, text="Положение", padding="5")
        position_frame.pack(fill=tk.X, pady=5)
        
        self.x_label = ttk.Label(position_frame, text="Дальность: 0.00 м", 
                               font=("Arial", 10))
        self.x_label.pack(anchor=tk.W)
        
        self.y_label = ttk.Label(position_frame, text="Высота: 0.00 м", 
                               font=("Arial", 10))
        self.y_label.pack(anchor=tk.W)
        
        # Время и ускорение
        time_frame = ttk.LabelFrame(right_frame, text="Время и ускорение", padding="5")
        time_frame.pack(fill=tk.X, pady=5)
        
        self.time_label = ttk.Label(time_frame, text="Время: 0.00 с", 
                                   font=("Arial", 10))
        self.time_label.pack(anchor=tk.W)
        
        self.g_current_label = ttk.Label(time_frame, text="g: 9.81 м/с²", 
                                       font=("Arial", 10))
        self.g_current_label.pack(anchor=tk.W)
        
        # График траектории
        trajectory_frame = ttk.LabelFrame(right_frame, text="Траектория", padding="5")
        trajectory_frame.pack(fill=tk.X, pady=5)
        
        self.show_trajectory_var = tk.BooleanVar(value=True)
        ttk.Checkbutton(trajectory_frame, text="Показывать траекторию", 
                       variable=self.show_trajectory_var,
                       command=self.draw_scene).pack(anchor=tk.W)
        
        self.show_grid_var = tk.BooleanVar(value=True)
        ttk.Checkbutton(trajectory_frame, text="Показывать сетку", 
                       variable=self.show_grid_var,
                       command=self.draw_scene).pack(anchor=tk.W)
        
        # Информация о полете
        info_frame = ttk.LabelFrame(right_frame, text="Информация", padding="5")
        info_frame.pack(fill=tk.X, pady=5)
        
        info_text = """Баллистическое движение:
• Без учета сопротивления
• Постоянное ускорение g
• Параболическая траектория
• Vx = const, Vy = V0y - gt"""
        
        info_label = ttk.Label(info_frame, text=info_text, font=("Arial", 8),
                              justify=tk.LEFT)
        info_label.pack(anchor=tk.W)
    
    def on_velocity_change(self, event=None):
        self.initial_velocity = self.velocity_var.get()
        self.velocity_value.config(text=f"{self.initial_velocity:.1f} м/с")
        self.calculate_trajectory()
        self.draw_scene()
    
    def on_angle_change(self, event=None):
        self.launch_angle = self.angle_var.get()
        self.angle_value.config(text=f"{self.launch_angle:.1f}°")
        self.calculate_trajectory()
        self.draw_scene()
    
    def on_height_change(self, event=None):
        self.initial_height = self.height_var.get()
        self.height_value.config(text=f"{self.initial_height:.1f} м")
        self.calculate_trajectory()
        self.draw_scene()
    
    def on_gravity_change(self, event=None):
        self.gravity = self.gravity_var.get()
        self.gravity_value.config(text=f"{self.gravity:.1f} м/с²")
        self.calculate_trajectory()
        self.draw_scene()
    
    def calculate_trajectory(self):
        """Расчет теоретической траектории"""
        # Преобразование угла в радианы
        angle_rad = math.radians(self.launch_angle)
        
        # Начальные скорости
        v0x = self.initial_velocity * math.cos(angle_rad)
        v0y = self.initial_velocity * math.sin(angle_rad)
        
        # Время полета
        if self.initial_height == 0:
            self.total_time = 2 * v0y / self.gravity
        else:
            # Квадратное уравнение для времени полета
            a = self.gravity / 2
            b = -v0y
            c = -self.initial_height
            discriminant = b**2 - 4*a*c
            if discriminant >= 0:
                self.total_time = (-b + math.sqrt(discriminant)) / (2*a)
            else:
                self.total_time = 0
        
        # Максимальная высота
        if self.initial_height == 0:
            self.max_height = (v0y**2) / (2 * self.gravity)
        else:
            time_to_max = v0y / self.gravity
            self.max_height = self.initial_height + v0y * time_to_max - 0.5 * self.gravity * time_to_max**2
        
        # Дальность полета
        self.max_range = v0x * self.total_time
        
        # Обновление теоретических значений
        self.max_height_label.config(text=f"Макс. высота: {self.max_height:.1f} м")
        self.max_range_label.config(text=f"Дальность полета: {self.max_range:.1f} м")
        self.total_time_label.config(text=f"Время полета: {self.total_time:.1f} с")
        
        # Расчет точек траектории для отрисовки
        self.trajectory = []
        if self.total_time > 0:
            time_step = self.total_time / 100
            
            for t in range(101):
                current_time = t * time_step
                x = v0x * current_time
                y = self.initial_height + v0y * current_time - 0.5 * self.gravity * current_time**2
                
                if y >= 0:  # Только пока объект над землей
                    self.trajectory.append((x, y))
    
    def start_simulation(self):
        if not self.simulating:
            self.simulating = True
            self.time_elapsed = 0
            self.start_button.config(state=tk.DISABLED)
            self.stop_button.config(state=tk.NORMAL)
            
            # Запускаем в основном потоке для избежания проблем с Tkinter
            self.root.after(10, self.run_simulation_step)
    
    def stop_simulation(self):
        self.simulating = False
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)
    
    def reset_simulation(self):
        self.simulating = False
        self.time_elapsed = 0
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)
        self.update_current_values(0, self.initial_height, 0, 0, 0)
        self.draw_scene()
    
    def run_simulation_step(self):
        """Выполнение одного шага симуляции"""
        if not self.simulating:
            return
            
        angle_rad = math.radians(self.launch_angle)
        v0x = self.initial_velocity * math.cos(angle_rad)
        v0y = self.initial_velocity * math.sin(angle_rad)
        
        time_step = 0.05  # 50 мс
        
        if self.time_elapsed <= self.total_time:
            # Расчет текущего положения
            x = v0x * self.time_elapsed
            y = self.initial_height + v0y * self.time_elapsed - 0.5 * self.gravity * self.time_elapsed**2
            
            # Расчет текущих скоростей
            vy = v0y - self.gravity * self.time_elapsed
            v = math.sqrt(v0x**2 + vy**2)
            
            # Обновление интерфейса
            self.update_current_values(x, y, v0x, vy, v)
            
            # Отрисовка
            self.draw_scene_with_object(x, y)
            
            self.time_elapsed += time_step
            self.root.after(50, self.run_simulation_step)  # 20 FPS
        else:
            self.simulating = False
            self.start_button.config(state=tk.NORMAL)
            self.stop_button.config(state=tk.DISABLED)
    
    def update_current_values(self, x, y, vx, vy, v):
        """Обновление текущих значений на правой панели"""
        self.vx_label.config(text=f"Vx: {vx:.2f} м/с")
        self.vy_label.config(text=f"Vy: {vy:.2f} м/с")
        self.v_label.config(text=f"V: {v:.2f} м/с")
        
        self.x_label.config(text=f"Дальность: {x:.2f} м")
        self.y_label.config(text=f"Высота: {y:.2f} м")
        
        self.time_label.config(text=f"Время: {self.time_elapsed:.2f} с")
        self.g_current_label.config(text=f"g: {self.gravity:.2f} м/с²")
    
    def draw_scene(self):
        """Отрисовка статической сцены"""
        self.draw_scene_with_object(None, None)
    
    def draw_scene_with_object(self, current_x=None, current_y=None):
        """Отрисовка сцены с текущим положением объекта"""
        self.canvas.delete("all")
        
        canvas_width = self.canvas.winfo_width()
        canvas_height = self.canvas.winfo_height()
        
        if canvas_width <= 1 or canvas_height <= 1:
            return
        
        # Автоматическое масштабирование
        if self.trajectory:
            max_x = max(point[0] for point in self.trajectory)
            max_y = max(point[1] for point in self.trajectory)
        else:
            max_x = 100
            max_y = 100
        
        scale_x = (canvas_width - 100) / max_x if max_x > 0 else 1
        scale_y = (canvas_height - 100) / max_y if max_y > 0 else 1
        self.scale = min(scale_x, scale_y) * 0.9
        
        # Земля
        ground_y = canvas_height - 50
        self.canvas.create_rectangle(0, ground_y, canvas_width, canvas_height, 
                                   fill="brown", outline="")
        
        # Сетка
        if self.show_grid_var.get():
            self.draw_grid(canvas_width, canvas_height, ground_y)
        
        # Траектория
        if self.show_trajectory_var.get() and self.trajectory:
            points = []
            for x, y in self.trajectory:
                screen_x = 50 + x * self.scale
                screen_y = ground_y - y * self.scale
                points.extend([screen_x, screen_y])
            
            if len(points) >= 4:
                self.canvas.create_line(points, fill="red", width=2, dash=(4, 2))
        
        # Текущее положение объекта
        if current_x is not None and current_y is not None and current_y >= 0:
            screen_x = 50 + current_x * self.scale
            screen_y = ground_y - current_y * self.scale
            
            # Объект
            self.canvas.create_oval(screen_x - 8, screen_y - 8, 
                                  screen_x + 8, screen_y + 8, 
                                  fill="blue", outline="darkblue", width=2)
            
            # Вектор скорости
            angle_rad = math.radians(self.launch_angle)
            v0x = self.initial_velocity * math.cos(angle_rad)
            v0y = self.initial_velocity * math.sin(angle_rad)
            vy = v0y - self.gravity * self.time_elapsed
            
            # Масштабирование вектора скорости для визуализации
            vector_scale = 0.5
            end_x = screen_x + v0x * vector_scale
            end_y = screen_y - vy * vector_scale
            
            self.canvas.create_line(screen_x, screen_y, end_x, end_y, 
                                  arrow=tk.LAST, fill="green", width=2)
        
        # Подписи осей
        self.canvas.create_text(30, ground_y - 20, text="y", font=("Arial", 12, "bold"), angle=90)
        self.canvas.create_text(canvas_width - 30, ground_y + 20, text="x", font=("Arial", 12, "bold"))
        
        # Шкала
        self.canvas.create_text(50, 20, 
                              text=f"Масштаб: 1px = {1/self.scale:.2f}м", 
                              font=("Arial", 10), anchor=tk.W)
    
    def draw_grid(self, width, height, ground_y):
        """Отрисовка сетки"""
        grid_step_meters = 10  # метров между линиями сетки
        grid_step_pixels = grid_step_meters * self.scale
        
        # Вертикальные линии
        x = 50
        while x < width:
            self.canvas.create_line(x, 30, x, ground_y, fill="gray", dash=(2, 2))
            meter_value = (x - 50) / self.scale
            self.canvas.create_text(x, ground_y + 15, text=f"{meter_value:.0f}", 
                                  font=("Arial", 8))
            x += grid_step_pixels
        
        # Горизонтальные линии
        y = ground_y
        line_count = 0
        while y > 30 and line_count < 20:
            self.canvas.create_line(50, y, width - 50, y, fill="gray", dash=(2, 2))
            meter_value = (ground_y - y) / self.scale
            self.canvas.create_text(30, y, text=f"{meter_value:.0f}", 
                                  font=("Arial", 8))
            y -= grid_step_pixels
            line_count += 1

def main():
    root = tk.Tk()
    app = BallisticMotion(root)
    root.mainloop()

if __name__ == "__main__":
    main()
